/**
 * Copyright (c) 2021-Now http://faasadmin.com All rights reserved.
 * No deletion without permission, or be held responsible to law.
 */

package com.faasadmin.faas.services.lessee.service.lessee.impl;

import cn.hutool.core.util.ObjectUtil;
import com.faasadmin.faas.services.lessee.convert.lessee.SaasLesseeConvert;
import com.faasadmin.faas.services.lessee.dal.dataobject.lessee.SaasLesseeDO;
import com.faasadmin.faas.services.lessee.dal.mysql.lessee.SaasLesseeMapper;
import com.faasadmin.faas.services.lessee.enums.SaasErrorCodeConstants;
import com.faasadmin.faas.services.lessee.vo.lessee.SaasLesseeCreateReqVO;
import com.faasadmin.faas.services.lessee.vo.lessee.SaasLesseeUpdateReqVO;
import com.faasadmin.faas.services.lessee.service.lessee.SaasLesseeService;
import com.faasadmin.framework.common.enums.CommonStatusEnum;
import com.faasadmin.framework.common.exception.util.ServiceExceptionUtil;
import com.faasadmin.framework.common.utils.DateUtils;
import com.faasadmin.framework.common.utils.ObjectUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.validation.annotation.Validated;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;

import static com.faasadmin.framework.common.exception.util.ServiceExceptionUtil.exception;

/**
 * 租户 Service 实现类
 *
 * @author mszf
 */
@Slf4j
@Service
@Validated
public class SaasLesseeServiceImpl implements SaasLesseeService {

    @Resource
    private SaasLesseeMapper lesseeMapper;

    @Override
    public Long createLessee(SaasLesseeCreateReqVO createReqVO) {
        // 插入
        SaasLesseeDO lessee = SaasLesseeConvert.INSTANCE.convert(createReqVO);
        lesseeMapper.insert(lessee);
        // 返回
        return lessee.getId();
    }

    @Override
    public void updateLessee(SaasLesseeUpdateReqVO updateReqVO) {
        // 校验存在
        this.validateLesseeExists(updateReqVO.getId());
        // 更新
        SaasLesseeDO updateObj = SaasLesseeConvert.INSTANCE.convert(updateReqVO);
        lesseeMapper.updateById(updateObj);
    }

    @Override
    public void deleteLessee(Long id) {
        // 校验存在
        this.validateLesseeExists(id);
        // 删除
        lesseeMapper.deleteById(id);
    }

    private void validateLesseeExists(Long id) {
        if (lesseeMapper.selectById(id) == null) {
            throw ServiceExceptionUtil.exception(SaasErrorCodeConstants.LESSEE_NOT_EXISTS);
        }
    }

    @Override
    public SaasLesseeDO getLessee(Long id) {
        return lesseeMapper.selectById(id);
    }

    @Override
    public List<SaasLesseeDO> getLesseeList(Collection<Long> ids) {
        return lesseeMapper.selectBatchIds(ids);
    }

    @Override
    public SaasLesseeDO checkLessee(Long lesseeId) {
        //判断是否过期
        SaasLesseeDO lessee = getLessee(lesseeId);
        if (ObjectUtil.isEmpty(lessee)) {
            log.error("该租户不存在,请联系客服!");
            throw ServiceExceptionUtil.exception(SaasErrorCodeConstants.LESSEE_NOT_EXISTS);
        }
        if (DateUtils.compare(DateUtils.date(), lessee.getEndTime()) > 0) {
            log.error("该租户已过期,请联系客服!");
            throw ServiceExceptionUtil.exception(SaasErrorCodeConstants.LESSEE_EXPIRED);
        }
        String status = lessee.getStatus();
        if (ObjectUtils.equals(CommonStatusEnum.DISABLE.getStatus(), status)) {
            log.error("该租户被停用,请联系客服!");
            throw ServiceExceptionUtil.exception(SaasErrorCodeConstants.LESSEE_DISABLE);
        }
        return lessee;
    }

}
