/**
 * Copyright (c) 2021-Now http://faasadmin.com All rights reserved.
 * No deletion without permission, or be held responsible to law.
 */

package com.faasadmin.faas.services.lessee.service.lesseeConfig.impl;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.ClassUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.faasadmin.faas.services.lessee.convert.lesseeConfig.SaasLesseeConfigConvert;
import com.faasadmin.faas.services.lessee.dal.dataobject.lesseeConfig.SaasLesseeConfigDO;
import com.faasadmin.faas.services.lessee.dal.mysql.lesseeConfig.SaasLesseeConfigMapper;
import com.faasadmin.faas.services.lessee.enums.SaasErrorCodeConstants;
import com.faasadmin.faas.services.lessee.vo.lesseeConfig.*;
import com.faasadmin.faas.services.lessee.service.lesseeConfig.SaasLesseeConfigService;
import com.faasadmin.framework.common.exception.ServiceException;
import com.faasadmin.framework.common.exception.util.ServiceExceptionUtil;
import com.faasadmin.framework.common.pojo.PageResult;
import com.faasadmin.faas.services.lessee.vo.lesseeConfig.*;
import org.springframework.stereotype.Service;
import org.springframework.validation.annotation.Validated;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;

import static com.faasadmin.framework.common.constant.Constants.STR_ZERO;
import static com.faasadmin.framework.common.exception.util.ServiceExceptionUtil.exception;

/**
 * 租户配置 Service 实现类
 *
 * @author faasadmin
 */
@Service
@Validated
public class SaasLesseeConfigServiceImpl extends ServiceImpl<SaasLesseeConfigMapper, SaasLesseeConfigDO> implements SaasLesseeConfigService {

    @Resource
    private SaasLesseeConfigMapper lesseeConfigMapper;

    @Override
    public Long createLesseeConfig(SaasLesseeConfigCreateReqVO createReqVO) {
        // 校验正确性
        checkCreateOrUpdate(null, createReqVO.getLesseeId(), createReqVO.getKey());
        // 插入
        SaasLesseeConfigDO lesseeConfig = SaasLesseeConfigConvert.INSTANCE.convert(createReqVO);
        lesseeConfigMapper.insert(lesseeConfig);
        // 返回
        return lesseeConfig.getId();
    }

    private void checkCreateOrUpdate(Long id, Long lesseeId, String key) {
        // 校验自己存在
        checkConfigExists(id);
        // 校验参数配置 key 的唯一性
        checkConfigKeyUnique(id, lesseeId, key);
    }

    public SaasLesseeConfigDO checkConfigExists(Long id) {
        if (id == null) {
            return null;
        }
        SaasLesseeConfigDO config = lesseeConfigMapper.selectById(id);
        if (config == null) {
            throw ServiceExceptionUtil.exception(SaasErrorCodeConstants.LESSEE_CONFIG_NOT_EXISTS);
        }
        return config;
    }

    public void checkConfigKeyUnique(Long id, Long lesseeId, String key) {
        SaasLesseeConfigDO config = lesseeConfigMapper.selectByKey(lesseeId, key);
        if (config == null) {
            return;
        }
        // 如果 id 为空，说明不用比较是否为相同 id 的参数配置
        if (id == null) {
            throw ServiceExceptionUtil.exception(SaasErrorCodeConstants.LESSEE_CONFIG_NOT_DUPLICATE);
        }
        if (!config.getId().equals(id)) {
            throw ServiceExceptionUtil.exception(SaasErrorCodeConstants.LESSEE_CONFIG_NOT_DUPLICATE);
        }
    }

    @Override
    public Long createLesseeConfig(SaasLesseeJsonConfigCreateReqVO createReqVO) {
        // 校验正确性
        checkCreateOrUpdate(null, createReqVO.getLesseeId(), createReqVO.getKey());
        // 插入
        SaasLesseeConfigDO lesseeConfig = SaasLesseeConfigConvert.INSTANCE.convert(createReqVO);
        lesseeConfigMapper.insert(lesseeConfig);
        // 返回
        return lesseeConfig.getId();
    }

    @Override
    public void updateLesseeConfig(SaasLesseeConfigUpdateReqVO updateReqVO) {
        // 校验正确性
        checkCreateOrUpdate(Convert.toLong(updateReqVO.getId()), updateReqVO.getLesseeId(), null); // 不允许更新 key
        // 更新
        SaasLesseeConfigDO updateObj = SaasLesseeConfigConvert.INSTANCE.convert(updateReqVO);
        lesseeConfigMapper.updateById(updateObj);
    }

    @Override
    public void updateLesseeConfigByKey(SaasLesseeConfigUpdateReqVO updateReqVO) {
        UpdateWrapper<SaasLesseeConfigDO> updateWrapper = new UpdateWrapper<SaasLesseeConfigDO>();
        updateWrapper.eq("`key`", updateReqVO.getKey());
        updateWrapper.eq("lessee_id", updateReqVO.getLesseeId());
        updateWrapper.set("value", updateReqVO.getValue());
        update(null, updateWrapper);
    }

    @Override
    public void updateLesseeJsonConfig(SaasLesseeJsonConfigUpdateReqVO updateReqVO) {
        // 校验正确性
        checkCreateOrUpdate(Convert.toLong(updateReqVO.getId()), updateReqVO.getLesseeId(), null); // 不允许更新 key
        // 更新
        SaasLesseeConfigDO updateObj = SaasLesseeConfigConvert.INSTANCE.convert(updateReqVO);
        lesseeConfigMapper.updateById(updateObj);
    }

    @Override
    public void updateLesseeConfig(SaasLesseeConfigDO updateReqVO) {
        // 校验正确性
        checkCreateOrUpdate(Convert.toLong(updateReqVO.getId()), updateReqVO.getLesseeId(), null); // 不允许更新 key
        // 更新
        lesseeConfigMapper.updateById(updateReqVO);
    }

    @Override
    public void deleteLesseeConfig(Long id) {
        // 校验存在
        this.validateLesseeConfigExists(id);
        // 删除
        lesseeConfigMapper.deleteById(id);
    }

    private void validateLesseeConfigExists(Long id) {
        if (lesseeConfigMapper.selectById(id) == null) {
            throw ServiceExceptionUtil.exception(SaasErrorCodeConstants.LESSEE_CONFIG_NOT_EXISTS);
        }
    }

    @Override
    public SaasLesseeConfigDO getLesseeConfig(Long id) {
        return lesseeConfigMapper.selectById(id);
    }

    @Override
    public <T> T getLesseeConfigObject(String key, Class<T> clazz, Long lesseeId) {
        SaasLesseeConfigDO infConfigDO = getLesseeConfigByKey(key, lesseeId);
        if (ObjectUtil.isEmpty(infConfigDO)) {
            return null;
        }
        String value = infConfigDO.getValue();
        //判断获取值,是否转换Bean对象
        if (!StrUtil.isBlankOrUndefined(value) && JSONUtil.isJson(value)
                && !ClassUtil.equals(clazz, "String", false)) {
            return JSONUtil.toBean(value, clazz);
        }
        //String 转换
        if (ClassUtil.equals(clazz, "String", false)) {
            return Convert.convert(clazz, value);
        }
        try {
            return clazz.newInstance();
        } catch (Exception e) {
            throw new ServiceException("获取参数失败");
        }
    }

    @Override
    public SaasLesseeConfigDO getLesseeConfigByKey(String key, Long lesseeId,String appId) {
        QueryWrapper<SaasLesseeConfigDO> queryWrapper = new QueryWrapper();
        queryWrapper.eq("`key`", key);
        queryWrapper.eq("lessee_id", lesseeId);
        queryWrapper.eq("app_id", appId);
        return lesseeConfigMapper.selectOne(queryWrapper);
    }

    @Override
    public SaasLesseeConfigDO getLesseeConfigByKey(String key, Long lesseeId) {
        QueryWrapper<SaasLesseeConfigDO> queryWrapper = new QueryWrapper();
        queryWrapper.eq("`key`", key);
        queryWrapper.eq("lessee_id", lesseeId);
        return lesseeConfigMapper.selectOne(queryWrapper);
    }

    @Override
    public List<SaasLesseeConfigDO> getLesseeConfigList(Collection<Long> ids) {
        return lesseeConfigMapper.selectBatchIds(ids);
    }

    @Override
    public PageResult<SaasLesseeConfigDO> getLesseeConfigPage(SaasLesseeConfigPageReqVO pageReqVO) {
        return lesseeConfigMapper.selectPage(pageReqVO);
    }

    @Override
    public List<SaasLesseeConfigDO> getLesseeConfigList(SaasLesseeConfigExportReqVO exportReqVO) {
        return lesseeConfigMapper.selectList(exportReqVO);
    }

    @Override
    public String selectLesseeConfigByKeyToStr(Long lesseeId, String appName, String keys, String def) {
        QueryWrapper<SaasLesseeConfigDO> queryWrapper = new QueryWrapper<SaasLesseeConfigDO>().eq("lessee_id", lesseeId).eq("app_name", appName).eq("`key`", keys);
        SaasLesseeConfigDO saasLesseeConfigDO = lesseeConfigMapper.selectOne(queryWrapper);
        if (ObjectUtil.isEmpty(saasLesseeConfigDO)) {
            return def;
        }
        return saasLesseeConfigDO.getValue();
    }

    @Override
    public Integer selectLesseeConfigByKeyToInt(Long lesseeId, String appName, String key) {
        return Convert.toInt(selectLesseeConfigByKeyToStr(lesseeId, appName, key, STR_ZERO));
    }

    @Override
    public Boolean selectLesseeConfigByKeyToBool(Long lesseeId, String appName, String key) {
        return Convert.toBool(selectLesseeConfigByKeyToStr(lesseeId, appName, key, STR_ZERO));
    }

}
