/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.lucene.analysis.payloads;

import java.io.IOException;
import java.util.Objects;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.PayloadAttribute;
import org.apache.lucene.analysis.tokenattributes.TypeAttribute;
import org.apache.lucene.util.BytesRef;

/**
 * Assigns a payload to a token based on the {@link
 * org.apache.lucene.analysis.tokenattributes.TypeAttribute}
 */
public class NumericPayloadTokenFilter extends TokenFilter {

  private final String typeMatch;
  private final BytesRef thePayload;

  private final PayloadAttribute payloadAtt = addAttribute(PayloadAttribute.class);
  private final TypeAttribute typeAtt = addAttribute(TypeAttribute.class);

  public NumericPayloadTokenFilter(TokenStream input, float payload, String typeMatch) {
    super(input);
    this.typeMatch = Objects.requireNonNull(typeMatch, "typeMatch");
    // Need to encode the payload
    this.thePayload = new BytesRef(PayloadHelper.encodeFloat(payload));
  }

  @Override
  public final boolean incrementToken() throws IOException {
    if (input.incrementToken()) {
      if (typeAtt.type().equals(typeMatch)) payloadAtt.setPayload(thePayload);
      return true;
    } else {
      return false;
    }
  }
}
