// @ts-check

window.monacoModule = window.monacoModule || {};

window.monacoModule.PromiseQueue = (function () {

  const { isNotNullOrUndefined } = window.monacoModule.helper;

  /**
   * A simple queue to which promise factories can be added. It makes sure the promises
   * are created / started in the order as they were added.
   * @template T
   * @implements IPromiseQueue<T>
   */
  class PromiseQueueImpl {
    constructor() {
      /** @type {QueueItem[]} */
      this.queue = [];
      /** @type {{resolve: PromiseCallback, reject: PromiseCallback}[]} */
      this.onDone = [];
    }

    /**
     * @param {PromiseFactory<T>} promiseFactory
     * @return {Promise<T>}
     */
    add(promiseFactory) {
      return this.addAll(promiseFactory)[0];
    }

    /**
     * @param {PromiseFactory<T>[]} promiseFactory
     * @return {Promise<T>[]}
     */
    addAll(...promiseFactory) {
      return promiseFactory
        .filter(isNotNullOrUndefined)
        .map(factory => new Promise((resolve, reject) => {
          this._addQueueItem(factory, resolve, reject);
        }));
    }

    /**
     *
     * @param {PromiseFactory<unknown>} factory
     * @param {PromiseCallback} resolve
     * @param {PromiseCallback} reject
     */
    _addQueueItem(factory, resolve, reject) {
      const wasEmpty = this.queue.length === 0;
      this.queue.push({
        factory: factory,
        resolve: resolve,
        reject: reject,
      });
      if (wasEmpty) {
        this._startQueue();
      }
    }

    _startQueue() {
      this._processQueue(this._peek());
    }

    onPromiseDone() {
      this._poll();
      this._processQueue(this._peek());
    }

    /**
     * @param {QueueItem} queueItem
     */
    _processQueue(queueItem) {
      if (queueItem) {
        const promise = PromiseQueueImpl._makePromise(queueItem.factory);
        promise
          .then(queueItem.resolve)
          .catch(queueItem.reject)
          .then(() => this.onPromiseDone());
      }
      else {
        this.onDone.forEach(({ resolve }) => resolve(undefined));
        this.onDone = [];
      }
    }

    /**
     * @return {QueueItem}
     */
    _poll() {
      return this.queue.shift();
    }

    /**
     * @return {QueueItem}
     */
    _peek() {
      return this.queue[0];
    }

    /**
     * @return {Promise<void>}
     */
    _allDone() {
      if (this.queue.length === 0) {
        return Promise.resolve();
      }
      else {
        return new Promise((resolve, reject) => {
          this.onDone.push({ resolve, reject });
        });
      }
    }

    /**
     * @param {PromiseFactory<unknown>} promiseFactory
     * @return {Promise<unknown>}
     */
    static _makePromise(promiseFactory) {
      try {
        const promise = promiseFactory();
        if (promise !== undefined) {
          return promise;
        }
      }
      catch (e) {
        console.error("[MonacoEditor] Could not create promise", e);
      }
      return Promise.resolve();
    }
  }

  return PromiseQueueImpl;
})();