package org.yunchen.gb.plugin.springsecurity.captcha

/**
 * Created by @Author:xiaopeng on 2018/4/21.
 */
import com.google.common.cache.CacheBuilder
import com.google.common.cache.CacheLoader
import com.google.common.cache.LoadingCache
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Service
import org.springframework.web.context.request.RequestContextHolder
import org.springframework.web.context.request.ServletRequestAttributes

import javax.annotation.PostConstruct
import java.util.concurrent.TimeUnit

@Service
@EnableConfigurationProperties([SpringSecurityCaptchaProperties.class])
class LoginAttemptCacheService {
    @Autowired
    SpringSecurityCaptchaProperties springSecurityCaptchaProperties

    private LoadingCache<String, Integer> attempts
    private Integer allowedNumberOfAttempts

    @PostConstruct
    public void init() {
        allowedNumberOfAttempts = springSecurityCaptchaProperties.allowedNumberOfAttempts.toInteger()
        Integer time = springSecurityCaptchaProperties.time.toInteger()
        attempts = CacheBuilder.newBuilder().expireAfterWrite(time, TimeUnit.MINUTES).build({0} as CacheLoader)
    }

    /**
     * Triggers on each unsuccessful login attempt and increases number of attempts
     */
    public void failLogin(String login) {
        int numberOfAttempts = attempts.get(login)
        numberOfAttempts++

        if (numberOfAttempts >= allowedNumberOfAttempts) {
            activateRecaptcha()
            //attempts.invalidate(login)
        } else {
            attempts.put(login, numberOfAttempts)
        }
    }

    /**
     * Triggers on each successful login attempt and resets number of attempts
     */
    public void loginSuccess(String login) {
        attempts.invalidate(login)
        deactivateRecaptcha()
    }

    private activateRecaptcha() {
        ((ServletRequestAttributes)RequestContextHolder?.getRequestAttributes())?.getRequest()?.session?.setAttribute("captchaForLogin",true)
    }

    private deactivateRecaptcha() {
        ((ServletRequestAttributes)RequestContextHolder?.getRequestAttributes())?.getRequest()?.session?.setAttribute("captchaForLogin",false)
    }
}
